// Form1.h
//
// This file implements a Form-derived class that is also
// a remote TPI.NET client. It demonstrates how to connect
// to a remote TLA and subscribe to events raised by TPI.NET.
/////////////////////////////////////////////////////////////

#include "StdAfx.h"
#using <mscorlib.dll>
#using <system.windows.forms.dll>
#using <system.drawing.dll>
#using <system.dll>

// Contains all the TPI.NET types and utility classes.
#using "TlaNetInterfaces.dll" 

#include "form1.h"

using namespace CPPRemoteClientSample;
using namespace System;
using namespace System::Collections;
using namespace System::Windows::Forms;
using namespace Tektronix::LogicAnalyzer::TpiNet;
using namespace Tektronix::LogicAnalyzer::Common;


// The constructor connects to the remote TLA application and 
// subscribes to two of the ITlaRunControl events.
Form1::Form1() :
	m_startCount( 0 ),
	m_endCount( 0 )
{
	InitializeComponent();
	
	String^ version = nullptr;
	
	// The assembly TlaNetInterfaces, which this project references,
	// contains a utility class named RemoteTlaSystem. It has a
	// Connect() static method which is used to set up a connection
	// to a remote TLA. To connect, an application configuration file
	// named ITlaSystemRemoteClient.config is used. This file contains
	// the information needed to configure this application to
	// communicate with a TLA running on the local host. The
	// configuration file can be modified to connect to a TLA
	// on a network.
	// NOTE: For the call to Connect() to succeed, the file
	// ITlaSystemRemoteClient.config must be in the same directory as
	// the application executable.
	try 
	{
		m_system = RemoteTlaSystem::Connect("ITlaSystemRemoteClient.config");
		version = m_system->SWVersion;
	}
	catch (...) 
	{
		// Failed to connect to the application. Show a warning message.
		MessageBox::Show("Could not connect to the TLA application. Make sure the TLA application is running before starting the client application. Also make sure the configuration file is located in the executable directory.");
		m_system = nullptr;
	}

	if (m_system == nullptr)
	{
		m_versionLabel->Text = "Could not connect to TLA application";
		m_runButton->Enabled = false;
		m_runButton->Text = "Cannot run TLA";
	}
	else
	{
		// Connection was successful. Now subscribe to TPI.NET events.
		SubscribeToTpiNetEvents();

		m_versionLabel->Text = String::Format("TLA Software Version is {0}", version);
		
		// Local events don't need shims.
		m_runButton->Click += gcnew EventHandler(this, &Form1::OnRunButton);
	}
}

// Destructor
Form1::~Form1()
{
	// Clean up any resources being used.
	UnsubscribeFromTpiNetEvents();
}

// Add TPI.NET event subscriptions.
void Form1::SubscribeToTpiNetEvents()
{
	if (m_system == nullptr) return;

	// Subscribe to ITlaRunControl.RunStarted.
	m_runStartHandler = EventRemoter::Create(gcnew EventHandler(this, &Form1::OnRunStarted));
	m_system->RunControl->RunStarted += m_runStartHandler;

	// Subscribe to ITlaRunControl.RunCompleted.
	m_runCompleteHandler = EventRemoter::Create(gcnew EventHandler(this, &Form1::OnRunCompleted));
	m_system->RunControl->RunCompleted += m_runCompleteHandler;
}

// Removes TPI.NET event subscriptions when the user exits the
// application. TPI.NET clients should always clean up their
// subscriptions before exiting.
void Form1::UnsubscribeFromTpiNetEvents()
{
	if (m_system == nullptr) return;

	// Unsubscribe from the run events.
	m_system->RunControl->RunStarted -= m_runStartHandler;
	m_system->RunControl->RunCompleted -= m_runCompleteHandler;

	m_runStartHandler = nullptr;
	m_runCompleteHandler = nullptr;
}

// This method initializes the size and position of controls 
// on the surface of the Form.
void Form1::InitializeComponent()
{
    m_versionLabel = gcnew Label;
    m_runButton = gcnew Button;
    m_label1 = gcnew Label;
    m_eventLabel = gcnew Label;
    SuspendLayout();
    // 
    // versionLabel
    // 
    m_versionLabel->Location = System::Drawing::Point(16, 24);
    m_versionLabel->Size = System::Drawing::Size(256, 32);
    m_versionLabel->TabIndex = 0;
    m_versionLabel->Text = "label1";
    // 
    // runButton
    // 
    m_runButton->Location = System::Drawing::Point(16, 64);
    m_runButton->Size = System::Drawing::Size(248, 32);
    m_runButton->TabIndex = 1;
    m_runButton->Text = "Run the TLA";
    // 
    // label1
    // 
    m_label1->Location = System::Drawing::Point(16, 120);
    m_label1->Size = System::Drawing::Size(256, 24);
    m_label1->TabIndex = 2;
    m_label1->Text = "Most Recent Run Event:";
    // 
    // eventLabel
    // 
    m_eventLabel->Location = System::Drawing::Point(16, 152);
    m_eventLabel->Name = "eventLabel";
    m_eventLabel->Size = System::Drawing::Size(256, 32);
    m_eventLabel->TabIndex = 3;
    m_eventLabel->Text = "None.";
    // 
    // CSRemoteClient
    // 
    AutoScaleBaseSize = System::Drawing::Size(6, 15);
    ClientSize = System::Drawing::Size(292, 208);
    Controls->Add(m_eventLabel);
    Controls->Add(m_label1);
    Controls->Add(m_runButton);
    Controls->Add(m_versionLabel);
    
    Text = "Remote TPI.NET Client";
    ResumeLayout(false);
}

// Event handler for the ITlaRunControl.RunStarted event.
void Form1::OnRunStarted(Object^ sender, EventArgs^ ea)
{
	// Asynchronous events from the TLA Application come in on worker threads.
	// In order to update the GUI controls, we must be on the UI thread.
	if (InvokeRequired)
	{
		// Make a call to the UI thread to output the results.
		array<Object^>^ args = { sender, ea };
		BeginInvoke(gcnew EventHandler(this, &Form1::OnRunStarted), args);
	}
	else
	{
		// We're already on the UI thread.  Output the results directly.
		m_startCount++;
		m_eventLabel->Text = String::Format("Acquisition {0} started.", m_startCount);
		m_runButton->Enabled = false;
	}
}

// Event handler for the ITlaRunControl.RunCompleted event.
void Form1::OnRunCompleted(Object^ sender, EventArgs^ ea)
{
	// Asynchronous events from the TLA Application come in on worker threads.
	// In order to update the GUI controls, we must be on the UI thread.
	if (InvokeRequired)
	{
		// Make a call to the UI thread to output the results.
		array<Object^>^ args = { sender, ea };
		BeginInvoke(gcnew EventHandler(this, &Form1::OnRunCompleted), args);
	}
	else
	{
		// We're already on the UI thread.  Output the results directly.
		m_endCount++;
		m_eventLabel->Text = String::Format("Acquisition {0} completed.", m_endCount);
		m_runButton->Enabled = true;
	}
}

// Starts a TLA acquistion when the user clicks the "Run the TLA" button.
void Form1::OnRunButton(Object^, EventArgs^)
{
	if (m_system == nullptr) return;

	m_system->RunControl->Run();
}


